﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/paytype")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class PayTypeController : BaseController
    {
        /// <summary>
        /// The payType services.
        /// </summary>
        private readonly IPayTypeService payServices;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public PayTypeController(IPayTypeService payServices, IAuditLogService auditLogServices)
        {
            this.payServices = payServices;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The fetch paytypes.
        /// </summary>
        /// <returns>
        /// The list of paytypes.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of paytypes.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAsync([FromBody] PayTypeModel model)
        {           
            model = (PayTypeModel)EmptyFilter.Handler(model);
            var response = await this.payServices.FetchAsync(model);
            return response == null ? this.ServerError() : this.Success(response);
        }

       
        /// <summary>
        /// Modifies the paytype status asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify")]
        public async Task<ActionResult> ModifyPayTypeStatusAsync([FromBody] PayTypeModel model, [FromHeader] LocationHeader header)
        {
            model = (PayTypeModel)EmptyFilter.Handler(model);
            var response = await this.payServices.ActivateOrDeactivatePayType(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.PayType,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = model.IsActive == true ? $@"<b>{model.ModifiedByName}</b> has Activated status of PayType <br/>name:<b> '{model.PayTypeName}'</b> on {DateTime.UtcNow.AddMinutes(330)}" : $@"<b>{model.ModifiedByName}</b> has Deactivated status of PayType<br/>name: <b>'{model.PayTypeName}'</b> on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);
        }

        /// <summary>
        /// The create async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> CreateAsync([FromBody] PayTypeModel model, [FromHeader] LocationHeader header)
        {
            model = (PayTypeModel)EmptyFilter.Handler(model);          
            var response = await this.payServices.InsertAsync(model);

            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.PayType,
                    LogDate = DateTime.Now,
                    LogFrom = (short)model.LoginRoleId,
                    LogDescription = $"<b>{model.CreatedByName}</b> has added Pay Type of <strong>{model.PayTypeName}</strong> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }
            return this.Success(response);
        }


        /// <summary>
        /// The update async.
        /// </summary>
        /// <param name="model"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] PayTypeModel model, [FromHeader] LocationHeader header)
        {
            model = (PayTypeModel)EmptyFilter.Handler(model);
            var response = await this.payServices.UpdateAsync(model);
            var auditLog = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.PayType,
                LogDate = DateTime.Now,
                LogFrom = (short)model.LoginRoleId,
                LogDescription = $"<b>{model.ModifiedByName}</b> has updated Pay Type of <strong>{model.PayTypeName}</strong> successfully.",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLog);
        
            return this.Success(response);
        }

    }

}